/**
 *
 *
 *  iRoom framework V2.0 <https://iroom.re>
 *
 *  By KRATZ Geoffrey AKA Jul6art AKA VanIllaSkype
 *  for VsWeb <https://vsweb.be>
 *
 *  https://vsweb.be
 *  admin@vsweb.be
 *
 *  Special thanks to Brynnlow
 *  for his contribution
 *
 *  It is free software; you can redistribute it and/or modify it under
 *  the terms of the GNU General Public License, either version 2
 *  of the License, or any later version.
 *
 *  For the full copyright and license information, please read the
 *  LICENSE.txt file that was distributed with this source code.
 *
 *  The flex one, in a flex world
 *
 */



module Iroom {
    declare let $: any;

    /**
     * Class Lightbox
     *
     * iMage and gallery lightbox
     */
    export class Lightbox {
        private options                     : LightboxOptions;
        private html                        : HTMLElement;
        private body                        : HTMLElement;
        private showcase                    : HTMLImageElement;
        private next                        : HTMLButtonElement;
        private prev                        : HTMLButtonElement;
        private close                       : HTMLButtonElement;
        private count                       : HTMLElement;
        private gallery                     : any;
        private lightboxBackground          : HTMLDivElement;
        index                               : number;
        private lightbox                    : any;


        /**
         * Lightbox constructor
         */
        constructor(options : LightboxOptions = null) {
            this.options                    = options;
            this.html                       = <HTMLElement>document.getElementsByTagName("html")[0];
            this.body                       = document.body;
            this.createLightBox();
            this.init();
        }


        /**
         * Create the divs required for the lightbox
         */
        createLightBox() {
            let lightboxBackground          = document.createElement('div');
            lightboxBackground.className    = 'lightboxBackground';
            lightboxBackground.classList.add('top');
            this.showcase                   = document.createElement('img');
            this.showcase.classList.add("showcase");
            this.next                       = document.createElement('button');
            this.next.className             = 'next';
            this.next.innerHTML             = '<i class="fa fa-chevron-right"></i>';
            this.prev                       = document.createElement('button');
            this.prev.className             = 'prev';
            this.prev.innerHTML             = '<i class="fa fa-chevron-left"></i>';
            this.close                      = document.createElement('button');
            this.close.innerHTML            = '<i class="fa fa-close"></i>';
            this.close.className            = 'close';
            this.count                      = document.createElement('p');
            this.count.innerHTML            = '';
            this.count.className            = 'count';
            lightboxBackground.appendChild(this.showcase);
            lightboxBackground.appendChild(this.close);
            lightboxBackground.appendChild(this.prev);
            lightboxBackground.appendChild(this.next);
            lightboxBackground.appendChild(this.count);
            this.lightboxBackground         = lightboxBackground;
            this.body.appendChild(lightboxBackground);
            this.lightbox                   = document.querySelectorAll('.lightbox');
        }


        /**
         * init events
         */
        init() {
            this.lightBoxEvent();
            this.prevEvent();
            this.nextEvent();
            this.closeEvent();
        }


        lightBoxEvent() {
            this.index              = null;
            for(let i = 0; i < this.lightbox.length; i++) {

                if(this.lightbox[i]) {
                    let parent = this.lightbox[i];

                    let arr         = Array.prototype.slice.call(this.lightbox); // Now it's an Array.
                    let index       = arr.indexOf(parent); // The index of your element :)

                    parent.addEventListener('click', (e) => {
                        this.gallery            = [];
                        this.count.innerHTML    = '';
                        if(parent != null) {
                            let element             = <HTMLImageElement>parent.querySelectorAll('img')[0];
                            element.setAttribute('index', String(index));
                            if(element != null) {
                                this.showcase.src       = element.src;
                                this.showcase.setAttribute('index', String(index));
                                if (typeof this.gallery === 'undefined' || !this.gallery.length) {
                                    this.gallery        = document.querySelectorAll(`.lightbox[data-gallery="${parent.getAttribute(`data-gallery`)}"] img`);
                                }

                                if (typeof this.index === 'undefined' || this.index  === null) {
                                    for(var i in this.gallery) {
                                        if(this.gallery[i]) {
                                            let image : HTMLElement  = this.gallery[i];
                                            if(this.getAttr(image, "index") &&
                                                this.getAttr(this.showcase, "index") &&
                                                this.getAttr(image, "index") ==
                                                this.getAttr(this.showcase, "index")) {
                                                this.index = parseInt(i);
                                            }
                                        }
                                    }
                                }

                                this.lightboxBackground.classList.add('active');
                                this.html.classList.add('showcase');
                                this.body.classList.add('showcase');
                                this.showcase.classList.add('active');
                                this.checkCommand(this.index);
                                this.enableArrowNavigation();
                            }
                        }
                    });
                }
            }
        }


        /**
         * Event when user pushes the prev button
         */
        prevEvent() {
            this.prev.addEventListener('click', (e) => {
                let img                     = <HTMLImageElement>this.gallery[--this.index];
                this.showcase.src           = img.src;
                this.checkCommand(this.index);
            });
        }


        /**
         * Event when user pushes the next button
         */
        nextEvent() {
            this.next.addEventListener('click', (e) => {
                let img                     = <HTMLImageElement>this.gallery[++this.index];
                this.showcase.src           = img.src;
                this.checkCommand(this.index);
            });
        }


        /**
         * Event when user pushes the close button
         */
        closeEvent() {
            this.close.addEventListener('click', (e) => {
                e.preventDefault();
                this.index                  = null;
                this.showcase.classList.remove('active');
                this.showcase.removeAttribute('index');
                this.lightboxBackground.classList.remove('active');
                this.html.classList.remove('showcase');
                this.body.classList.remove('showcase');
            });
        }


        /**
         * function that checks if there is a prev item and a next item
         *
         * if yes, it enable the navigation buttons
         *
         * @param index
         */
        checkCommand(index = null) {
            if(index > 0 && this.gallery.length > 1) {
                this.prev.classList.add('active');
            } else {
                this.prev.classList.remove('active');
            }

            if(index < this.gallery.length - 1) {
                this.next.classList.add('active');
            } else {
                this.next.classList.remove('active');
            }

            if(index >= 0 && this.gallery.length >= 1) {
                this.count.innerHTML = '<i class="fa fa-camera"></i> ' + (this.index + 1) + ' of ' + this.gallery.length;
            }
        }


        /**
         * function that allow or disallow navigation with the keyboard
         */
        enableArrowNavigation() {
            document.onkeydown = (e) => {
                switch (e.keyCode) {
                    case 37: // left key
                        e.preventDefault();
                        if(this.prev.classList.contains('active')) {
                            this.prev.click();
                        }
                        break;
                    case 39: // right arrow
                        e.preventDefault();
                        if(this.next.classList.contains('active')) {
                            this.next.click();
                        }
                        break;
                    case 27: // escape
                        e.preventDefault();
                        this.close.click();
                        break;
                }
            }
        }


        /**
         * Function to hack the IE BUG
         *
         * Undefiend function eobject.getAttribute()
         *
         * @param ele
         * @param attr
         * @returns {(function(string=): string)|string|null}
         */
        getAttr(ele, attr) {
            let result = (ele.getAttribute && ele.getAttribute(attr)) || null;
            if( !result ) {
                var attrs = ele.attributes;
                if(attrs != null) {
                    var length = attrs.length;
                    for(var i = 0; i < length; i++) {
                        if(attrs[i].nodeName === attr) {
                            result = attrs[i].nodeValue;
                        }
                    }
                }
            }
            return result;
        }
    }

    /**
     * interface Lightbox Options
     */
    interface LightboxOptions {
        duration : number;
    }
}